<?php


class WordReplacerAdmin {
    private $api;
    private $db;
    private $content_processor;
    private $article_generator;

    public function __construct(WordReplacerAPI $api, WordReplacerDB $db, WordReplacerContentProcessor $content_processor) {
        $this->api = $api;
        $this->db = $db;
        $this->content_processor = $content_processor;
        $this->article_generator = new WordReplacerArticleGenerator($api);
    }

    public function init() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('wp_ajax_get_word_link_table', [$this, 'ajax_get_word_link_table']);
        add_action('wp_ajax_verify_secret_key', [$this, 'ajax_verify_secret_key']);
        add_action('wp_ajax_replace_content', [$this, 'ajax_replace_content']);
        add_action('wp_ajax_logout_word_replacer', [$this, 'ajax_logout_word_replacer']);
        add_action('wp_ajax_generate_article', [$this, 'ajax_generate_article']);
        add_action('wp_ajax_publish_generated_article', [$this, 'ajax_publish_generated_article']);
        add_action('wp_ajax_get_article_form_information', [$this, 'ajax_get_article_form_information']);
    }

    public function add_admin_menu() {
        add_menu_page(
            'Adverlink', // Page title
            'Adverlink', // Menu title
            'manage_options', // Capability required to access
            'word-replacer', // Menu slug
            [$this, 'render_settings_page'], // Callback function to display the page
            'dashicons-admin-links', // Icon (you can change this to any dashicon you prefer)
            30 // Position in the menu (adjust as needed)
        );

        add_submenu_page(
            null, // Hidden from menu
            'Generate Article',
            'Generate Article',
            'manage_options',
            'word-replacer-article-generator',
            [$this, 'render_article_generator_page']
        );
    }

    public function enqueue_admin_scripts($hook) {
    // Check if we're on either the main plugin page or the article generator page
        if (!in_array($hook, ['toplevel_page_word-replacer', 'admin_page_word-replacer-article-generator'])) {
            return;
        }

    // Enqueue main admin CSS for both pages
    wp_enqueue_style(
        'word-replacer-admin-css',
        WORD_REPLACER_PLUGIN_URL . 'admin/css/admin.css',
        [],
        WORD_REPLACER_VERSION
    );

    // Ensure jQuery is loaded
    wp_enqueue_script('jquery');

    // Enqueue main admin JavaScript for both pages
    wp_enqueue_script(
        'word-replacer-admin-js',
        WORD_REPLACER_PLUGIN_URL . 'admin/js/admin.js',
        ['jquery'],
        WORD_REPLACER_VERSION . '.' . time(),
        true
    );

    wp_localize_script(
        'word-replacer-admin-js',
        'wordReplacerAjax',
        [
            'ajax_url' => admin_url('admin-ajax.php'),
            'admin_url' => admin_url('admin.php'),
            'nonce' => wp_create_nonce('word-replacer-nonce'),
            'plugin_url' => WORD_REPLACER_PLUGIN_URL
        ]
    );

    // Add article generator specific files only on the article generator page
    if ($hook === 'admin_page_word-replacer-article-generator') {
        // Enqueue article generator specific CSS - ADD THIS PART
        wp_enqueue_style(
            'word-replacer-article-generator-css',
            WORD_REPLACER_PLUGIN_URL . 'admin/css/article-generator.css',
            ['word-replacer-admin-css'],
            WORD_REPLACER_VERSION . '.' . time()
        );

        // Article generator JS (already in your code)
        wp_enqueue_script(
            'word-replacer-article-generator-js',
            WORD_REPLACER_PLUGIN_URL . 'admin/js/article-generator.js',
            ['jquery', 'word-replacer-admin-js'],
            WORD_REPLACER_VERSION . '.' . time(),
            true
        );

        wp_localize_script(
            'word-replacer-article-generator-js',
            'wordReplacerArticle',
            [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('word-replacer-article-nonce')
            ]
        );
    }
}

    public function render_settings_page() {
        $user_id = get_option('word_replacer_user_id');
        $secret_key = get_option('word_replacer_secret_key');
        $tables_created = $this->db->tables_exist();

        // Check plugin version
        $current_version = WORD_REPLACER_VERSION;
        $latest_version = $this->api->get_latest_version();
        $needs_update = 0;
        if ($latest_version && version_compare($latest_version, $current_version, '>')) {
            $needs_update = 1;
        }
        ?>
        <div class="wrap">
        <h1>Adverlink</h1>

        <!-- <?php if ($needs_update && $tables_created === true): ?>
            <div class="notice notice-error">
                <p><strong>Warning:</strong> Your plugin is outdated! Please update to the latest version.</p>
                
            </div>
            <?php endif; ?> -->

            <?php if ($tables_created === false): ?>
            <div class="notice notice-error">
                <p><strong>Error:</strong> Failed to create necessary database tables for the Adverlink plugin. Please check your database permissions and contact your system administrator.</p>
            </div>
            <?php elseif (!$user_id || !$secret_key): ?>
            <div id="secret-key-section">
                <label for="secret-key">Enter Secret Key:</label>
                <input type="text" id="secret-key" name="secret-key">
                <button id="verify-secret-key" class="button button-primary">Verify Secret Key</button>
            </div>
            <?php else: ?>
            <div id="settings-content">
                <h2>Word-Link Table</h2>
                <div id="word-link-table">Loading...</div>
                <div id="replacement-result"></div>
            </div>
            <?php endif; ?>

            <!-- Add this for debugging --
        </div>
        <?php
    }

    public function ajax_verify_secret_key() {
        // check_ajax_referer('word-replacer-nonce', 'nonce');

        $secret_key = sanitize_text_field($_POST['secret_key']);
        $user_id = $this->api->verify_secret_key($secret_key);

        if ($user_id) {
            update_option('word_replacer_user_id', $user_id);
            update_option('word_replacer_secret_key', $secret_key);
            wp_send_json_success(['user_id' => $user_id]);
        } else {
            wp_send_json_error('Invalid secret key or API response.');
        }
    }

    public function ajax_get_word_link_table() {
        try {
            // Verify nonce
            if (!check_ajax_referer('word-replacer-nonce', 'nonce', false)) {
                wp_send_json_error('Invalid security token.');
                return;
            }

            // Check user capabilities
            if (!current_user_can('manage_options')) {
                wp_send_json_error('Insufficient permissions.');
                return;
            }

            // Get and validate user data
            $user_id = get_option('word_replacer_user_id');
            $secret_key = get_option('word_replacer_secret_key');
            
            if (!$user_id || !$secret_key) {
                wp_send_json_error('Missing user credentials.');
                return;
            }

            // Get word list
            $word_and_link_list = $this->api->get_word_and_link($user_id, $secret_key);

            if (!$word_and_link_list || isset($word_and_link_list['error'])) {
                $error_message = isset($word_and_link_list['error']) ? $word_and_link_list['error'] : 'Unknown error';
                wp_send_json_error($error_message);
                return;
            }

            // Generate table HTML
            $table_html = $this->generate_word_link_table_html($word_and_link_list);
            wp_send_json_success($table_html);
            
        } catch (Exception $e) {
            wp_send_json_error('An unexpected error occurred.');
        }
    }
    
    

    public function ajax_logout_word_replacer() {
        error_log("Word: " . print_r('ok', true));
        check_ajax_referer('word-replacer-nonce', 'nonce');

        delete_option('word_replacer_user_id');
        delete_option('word_replacer_secret_key');

        wp_send_json_success();
    }

    private function generate_word_link_table_html($word_and_link_list) {
        // Add debugging
        error_log('Word and link list: ' . print_r($word_and_link_list, true));
        
        ob_start();
        ?>
        <div class="tablenav top">
            <div class="alignleft actions bulkactions">
                <button id="bulk-replace" class="button button-primary">
                    <span class="button-text">Replace All Checked</span>
                    <div class="button-spinner" id="bulk-replace-spinner"></div>
                </button>
                <button id="bulk-revert" class="button button-secondary">
                    <span class="button-text">Revert All Checked</span>
                    <div class="button-spinner" id="bulk-revert-spinner"></div>
                </button>
            </div>
        </div>
        <div id="word-replacer-notices" class="notice-container"></div>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th>Word</th>
                    <th>Link</th>
                    <th>Title</th>
                    <th>Type of link</th>
                    <th>Article URL</th>
                    <th>Status</th>
                    <th>Number of occurrences</th>
                    <th style="display: none;">Activate Word</th>
                    <th style="text-align: center;">Actions</th>
                    <th><input type="checkbox" id="select-all-words"></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($word_and_link_list as $index => $item): 
                    // Add debugging for each item
                    error_log('Processing item: ' . print_r($item, true));
                    
                    $occurrences = isset($item['occurences']) ? $item['occurences'] : 0;
                    $maxArticleOccurences = isset($item['maxArticleOccurences']) ? $item['maxArticleOccurences'] : 0;
                    $replacement_url = $item['targetURL'];
                    $is_active = $item['isActive'];
                    $replaced_keyword = isset($item['replacedKeyword']) ? $item['replacedKeyword'] : '';
                    $original_word = $item['word'];

                    $row_attributes = sprintf(
                        'data-word="%s" data-link="%s" data-title="%s" data-type-of-link="%s"',
                        esc_attr($item['word']),
                        esc_attr($item['link']),
                        esc_attr($item['title']),
                        esc_attr($item['typeOfLink'] ?? 'follow')
                    );
                ?>
                    <tr class="<?php echo $is_active ? 'word-replacer-active-word' : ''; ?>" 
                        <?php echo $row_attributes; ?>
                        data-is-active="<?php echo $is_active ? 'true' : 'false'; ?>"
                        data-replaced-keyword="<?php echo esc_attr($replaced_keyword); ?>"
                        data-wtl-id="<?php echo esc_attr($item['wtlId']); ?>">
                        <td>
                            <?php echo esc_html($original_word); ?>
                            <?php if (!empty($replaced_keyword) && $replaced_keyword !== $original_word): ?>
                                <div class="replaced-word">
                                    (<?php echo esc_html($replaced_keyword); ?>)
                                </div>
                            <?php endif; ?>
                        </td>
                        <td><a href="<?php echo esc_url($item['link']); ?>"><?php echo esc_url($item['link']); ?></a></td>
                        <td><?php echo esc_html($item['title']); ?></td>
                        <td><?php echo esc_html($item['typeOfLink']); ?></td>
                        <td>
                            <?php if ($replacement_url): ?>
                                <a href="<?php echo esc_url($replacement_url); ?>"><?php echo esc_html($replacement_url); ?></a>
                            <?php else: ?>
                                No replacement URL
                            <?php endif; ?>
                        </td>
                        <td><?php echo $is_active ? 'Active' : 'Inactive'; ?></td>
                        <td><?php echo esc_html($occurrences . '/' . $maxArticleOccurences); ?></td>
                        <td style="display: none;">
                            <input type="checkbox" name="activate_word[]" value="<?php echo $index; ?>" 
                                <?php checked($is_active); ?> class="word-replacer-activate-checkbox">
                        </td>
                        <td>
                            <div style="display: flex; flex-wrap: wrap; gap: 5px; align-items: center; justify-content: center;">
                                <button class="button button-primary word-replacer-replace-single" 
                                    data-index="<?php echo esc_attr($index); ?>"
                                    data-word="<?php echo esc_attr($item['word']); ?>"
                                    data-link="<?php echo esc_attr($item['link']); ?>"
                                    data-title="<?php echo esc_attr($item['title']); ?>"
                                    data-type-of-link="<?php echo esc_attr($item['typeOfLink']); ?>"
                                    <?php echo $is_active ? 'disabled' : ''; ?>>
                                    <span class="button-text">ACTIVATE</span>
                                    <div class="button-spinner"></div>
                                </button>
                                <button class="button button-secondary word-replacer-revert-single" 
                                    data-index="<?php echo esc_attr($index); ?>"
                                    data-word="<?php echo esc_attr($item['word']); ?>"
                                    data-link="<?php echo esc_attr($item['link']); ?>"
                                    data-title="<?php echo esc_attr($item['title']); ?>"
                                    data-type-of-link="<?php echo esc_attr($item['typeOfLink']); ?>"
                                    <?php echo !$is_active ? 'disabled' : ''; ?>>
                                    <span class="button-text">DEACTIVATE</span>
                                    <div class="button-spinner"></div>
                                </button>
                            </div>
                        </td>
                        <td>
                            <input type="checkbox" class="word-select" 
                                   data-index="<?php echo esc_attr($index); ?>"
                                   data-is-active="<?php echo $is_active ? 'true' : 'false'; ?>">
                        </td>
                    </tr>
                    <tr class="word-message-row" style="display: none;">
                        <td colspan="9">
                            <div class="word-message"></div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php
        return ob_get_clean();
    }
    
       
    public function ajax_replace_content() {
        check_ajax_referer('word-replacer-nonce', 'nonce');
    
        $user_id = get_option('word_replacer_user_id');
        $secret_key = get_option('word_replacer_secret_key');
        $word_and_link_list = $this->api->get_word_and_link($user_id, $secret_key);
    
        if (!$word_and_link_list) {
            wp_send_json_error('Failed to retrieve word-link data.');
        }
        
        $active_words = isset($_POST['active_words']) ? array_map('intval', $_POST['active_words']) : [];
        $is_bulk_revert = isset($_POST['is_bulk_revert']) && $_POST['is_bulk_revert'] === 'true';
        $results = [];

        $max_occurences_article = $this->api->get_max_occurences($user_id, $secret_key);

        // If it's a single word operation, only process that specific index
        $single_word_index = isset($_POST['single_word_index']) ? intval($_POST['single_word_index']) : null;
        
        foreach ($word_and_link_list as $index => $item) {
            // Skip if it's a single word operation and this isn't the target word
            if ($single_word_index !== null && $index !== $single_word_index) {
                continue;
            }
            
            // Skip if it's a bulk operation and this word isn't in the selected list
            if ($single_word_index === null && !in_array($index, $active_words)) {
                continue;
            }

            $WTLId = $item['wtlId'];
            $replacement_url = '';
            $word = $item['word'] ?? '';
            $was_active = $item['isActive'];
            
            // Determine if the word should be active based on operation type
            $is_active = $is_bulk_revert ? false : in_array($index, $active_words);

            $max_occurrences = 1;

            if ($is_active && !$was_active) {
                // Replace content for newly activated words
                $api_results = $this->api->call_search_api($word, parse_url(home_url(), PHP_URL_HOST));
                $content_replaced = false;
                
                if (empty($api_results)) {
                    // Adăugăm mesaj când nu există articole
                    $results[] = "No articles found containing the word '{$word}'. The word cannot be activated.";
                    continue;
                }
                
                if (is_array($api_results)) {
                    foreach ($api_results as $target_url) {
                        if (is_string($target_url)) {
                            $post_id = url_to_postid($target_url);
                            if ($post_id) {
                                $current_count = $this->db->get_replacement_url_count($target_url);
                                if ($current_count >= $max_occurences_article) 
                                {
                                    $results[] = [
                                        'word' => $word,
                                        'url' => $target_url,
                                        'status' => 'failed',
                                        'message' => "Maximum occurrences reached for word '$word' at URL '$target_url'. Current count: $current_count, Max allowed: $max_occurrences."
                                    ];
                                    continue;
                                }
                                $result = $this->content_processor->replace_content_for_post($post_id, $item);
                                if ($result['success']) {
                                    $count = $result['count'];
                                    $replaced_word = $result['replaced_word'] ?? '';
                                    $target_url = $result['target_url'] ?? '';
                                    $results[] = sprintf(
                                        "Content replaced for '%s' in post ID: %d. Occurrences: %d, target_url: %s, replaced_keyword: '%s'",
                                        $word,
                                        $post_id,
                                        $count,
                                        $target_url,
                                        $replaced_word
                                    );
                                    
                                    if($count != 0) {
                                        $replacement_url = $target_url;
                                        $content_replaced = true;
                                        // Update the item with new values
                                        $item['max_occurrences'] = $item['maxArticleOccurences'];
                                        $item['occurrences'] = $count;
                                        $item['replacement_url'] = $replacement_url ?? '';
                                        $item['replacedKeyword'] = $replaced_word;
                                        $word_and_link_list[$index] = $item;
                                        $update_result = $this->db->update_single_active_word($item);
                                        
                                        $dbWordUpdate = $this->api->update_word_to_link(
                                            $WTLId, 
                                            $replacement_url, 
                                            true,
                                            $replaced_word,
                                            $count
                                        );
                                        
                                        $setDateMin = $this->api->setUpdatedAtToMin($user_id, $secret_key, $word, $item['link'], $item['title']); 
                                        break;
                                    }
                                }
                            } else {
                                $results[] = "Could not find a valid post for the word '{$word}' at URL: {$target_url}";
                            }
                        }
                    }
                } else {
                    $results[] = "Invalid API response for word '{$word}'. Please try again.";
                }
                
                // Dacă nu s-a făcut nicio înlocuire după verificarea tuturor articolelor
                if (!$content_replaced) {
                    $results[] = "Could not activate '{$word}'. The word was not found in any suitable article.";
                }
            } elseif (!$is_active && $was_active) {
                // Revert content for deactivated words
                $post_id = url_to_postid($item['targetURL']);
                if (!empty($post_id)) {
                    $revert_success = true;
                    $result = $this->content_processor->revert_content_for_post($post_id, $word, $item['link']);
                    if (!$result['success']) {
                        $revert_success = false;
                        $results[] = "Failed to revert content for '{$word}' in post ID: {$post_id}";
                    } else {
                        $target_url = $result['target_url'] ?? '';
                        $results[] = "Content reverted for '{$word}' in post ID: {$post_id}, target_url: {$target_url}";
                    }
                    
                    if ($revert_success) {
                        $deactivate_result = $this->db->deactivate_single_word($word, $item['link']);
                        $dbWordUpdate = $this->api->update_word_to_link($WTLId, $replacement_url, false); 
                        $this->api->setUpdatedAtToMin($user_id, $secret_key, $word, $item['link'], $item['title']);
                        if ($deactivate_result === false) {
                            $results[] = "Failed to deactivate word '{$word}' in the database.";
                        } else {
                            $results[] = "Successfully deactivated word '{$word}' in the database.";
                        }
                    }
                }
            } elseif ($is_active && $was_active) {
                $count = $this->db->get_occurrences($word, $item['link']);
                $replacement_url = $item['targetURL'];
                $results[] = "'{$word}' was already active, no changes made";
            } else {
                // First, check if the word was used on the page
                $count = $this->db->get_occurrences($word, $item['link']);
                $replacement_url = $item['targetURL'];
            
                // Log or add debug information to track skipped conditions
                $debug_message = "Checking conditions for word '{$word}' on link '{$item['link']}':\n";
                
                if ($count == 0) {
                    $debug_message .= "The word '{$word}' has no occurrences on this page.\n";
                }
            
                // Check if the word is not activated for this link
                if (!$this->db->is_word_active($word, $item['link'])) {
                    $debug_message .= "The word '{$word}' is not activated for this link, so no action was taken.\n";
                }
            
                // Log additional conditions or assumptions
                $results[] = "Word '{$word}' remains inactive, no changes made. This may be due to:\n" . $debug_message;
            }
            
            
    
            // Update the item with new values
            //idk if used anymore
            $item['max_occurrences'] = $max_occurrences;

            $item['occurrences'] = 0;
            $item['replacement_url'] = '';
            $word_and_link_list[$index] = $item;
    
            // Debugging
            error_log('Updated word_and_link_list: ' . print_r($word_and_link_list, true));
        }
    
        // Update the active words in the database
        error_log('la final' . print_r($word_and_link_list, true));
        // Log the results of the replacement operation
        $log_results = array(
            'Source' => 'php-plugin',
            'Level' => 'Info',
            'Action' => 'replacement_results',
            'JsonData' => json_encode($results),
            'UserId' => $user_id,
            'AdditionalInfo' => 'Replacement process completed'
        );
        $this->api->postLog($log_results);

        //12.08.2024
        //$this->db->update_active_words($word_and_link_list, $active_words);
    
        wp_send_json_success(implode("<br>", $results));
    }
  
    public function render_article_generator_page() {
        // Verify the nonce first
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'word-replacer-nonce')) {
            wp_die('Unauthorized access. Please generate articles from the main dashboard.');
        }

        // Get the submitted data
        $word = isset($_POST['word']) ? sanitize_text_field($_POST['word']) : '';
        $link = isset($_POST['link']) ? esc_url_raw($_POST['link']) : '';
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
        $typeOfLink = isset($_POST['typeOfLink']) ? sanitize_text_field($_POST['typeOfLink']) : '';
        $wtlId = isset($_POST['wtlId']) ? intval($_POST['wtlId']) : 0;

        // If any required data is missing, redirect back to main page
        if (!$word || !$link || !$title || !$wtlId) {
            wp_redirect(admin_url('admin.php?page=word-replacer'));
            exit;
        }
        
        error_log('Article generator page data: ' . print_r([
            'word' => $word,
            'link' => $link,
            'title' => $title,
            'typeOfLink' => $typeOfLink,
            'wtlId' => $wtlId
        ], true));
        
        // Pass the data to the article generator
        $word_data = [
            'word' => $word,
            'link' => $link,
            'title' => $title,
            'typeOfLink' => $typeOfLink,
            'wtlId' => $wtlId
        ];
        
        $this->article_generator->render_article_form($word, $word_data);
    }

    public function ajax_generate_article() {
        check_ajax_referer('word-replacer-nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Insufficient permissions']);
            return;
        }

        $article_data = $_POST['article_data'] ?? null;
        
        // Log the received data
        error_log('Received article data in AJAX handler: ' . print_r($article_data, true));
        
        if (!$article_data) {
            error_log('No article data provided in AJAX request');
            wp_send_json_error(['error' => 'No article data provided']);
            return;
        }

        // Validate required fields
        $required_fields = ['keyword', 'wordCount', 'topic', 'language'];
        foreach ($required_fields as $field) {
            if (empty($article_data[$field])) {
                error_log("Missing required field: $field");
                wp_send_json_error(['error' => "Missing required field: $field"]);
                return;
            }
        }

        // Ensure wordCount is an integer
        $article_data['wordCount'] = intval($article_data['wordCount']);
        if ($article_data['wordCount'] < 2000 || $article_data['wordCount'] > 3000) {
            error_log("Invalid word count: {$article_data['wordCount']}");
            wp_send_json_error(['error' => 'Word count must be between 2000 and 3000']);
            return;
        }

        // Generate the article
        $result = $this->api->generate_article($article_data);
        
        // If generation successful, process the content to add links
        if (!isset($result['error']) && isset($result['content']['htmlContent'])) {
            // Create word data array for replacement
            $word_data = [
                'word' => $article_data['keyword'],
                'link' => $article_data['link'] ?? '',
                'title' => $article_data['title'] ?? '',
                'typeOfLink' => $article_data['typeOfLink'] ?? 'follow'
            ];

            // Process the content to add links using the public method
            $processed_content = $this->content_processor->process_article_content(
                $result['content']['htmlContent'],
                $word_data
            );

            // Update the content in the result
            if (isset($processed_content['new_content'])) {
                $result['content']['htmlContent'] = $processed_content['new_content'];
            }
        }
        
        // Log the API result
        error_log('API generation result: ' . print_r($result, true));
        
        if (isset($result['error'])) {
            wp_send_json_error(['error' => $result['error']]);
            return;
        }

        wp_send_json_success($result);
    }

    public function ajax_publish_generated_article() {
        check_ajax_referer('word-replacer-nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['error' => 'Insufficient permissions']);
            return;
        }

        $post_data = [
            'post_title'   => sanitize_text_field($_POST['title']),
            'post_content' => wp_kses_post($_POST['content']),
            'post_status'  => 'publish', // Temporar setăm ca publish
            'post_type'    => 'post'
        ];

        $post_id = wp_insert_post($post_data);

        if (is_wp_error($post_id)) {
            wp_send_json_error(['error' => $post_id->get_error_message()]);
            return;
        }

        // Obținem URL-ul permanent
        $post_url = get_permalink($post_id);
        
        // Actualizăm înapoi la draft
        wp_update_post([
            'ID' => $post_id,
            'post_status' => 'draft'
        ]);

        // Get the WTL ID from the request
        $wtl_id = isset($_POST['wtlId']) ? intval($_POST['wtlId']) : 0;
        
        if ($wtl_id) {
            // Update the WordToLink status
            $update_result = $this->api->update_word_to_link(
                $wtl_id,      // WTL ID
                $post_url,    // Target URL (acum va fi URL-ul permanent)
                true,         // Set as active
                '',          // No replaced keyword for this case
                1            // Single occurrence
            );

            if (isset($update_result['error'])) {
                error_log('Failed to update WordToLink: ' . print_r($update_result['error'], true));
            }
        }

        wp_send_json_success([
            'message' => 'Article saved as draft successfully',
            'editUrl' => get_edit_post_link($post_id, 'url')
        ]);
    }

    public function ajax_get_article_form_information() {
        try {
            // Verify nonce
            if (!check_ajax_referer('word-replacer-nonce', 'nonce', false)) {
                error_log('Word Replacer: Nonce verification failed');
                wp_send_json_error(['message' => 'Security check failed']);
                return;
            }

            if (!current_user_can('manage_options')) {
                error_log('Word Replacer: Insufficient permissions');
                wp_send_json_error(['message' => 'Insufficient permissions']);
                return;
            }
            
            $wtl_id = isset($_POST['wtlId']) ? intval($_POST['wtlId']) : 0;
            $secret_key = get_option('word_replacer_secret_key');
            
            error_log('Word Replacer: Attempting to get form info with WTL ID: ' . $wtl_id);
            
            if (!$wtl_id || !$secret_key) {
                error_log('Word Replacer: Missing required data - WTL ID: ' . $wtl_id);
                wp_send_json_error(['message' => 'Missing required data']);
                return;
            }
            
            $form_info = $this->api->get_article_form_information($wtl_id, $secret_key);
            error_log('Word Replacer: API response: ' . print_r($form_info, true));
            
            if ($form_info === false) {
                error_log('Word Replacer: Failed to get form information from API');
                wp_send_json_error(['message' => 'Failed to get form information']);
                return;
            }
            
            wp_send_json_success($form_info);
            
        } catch (Exception $e) {
            error_log('Word Replacer Error: ' . $e->getMessage());
            wp_send_json_error(['message' => 'An unexpected error occurred']);
        }
    }

    public function display_word_and_link_list($word_and_link_list) {
        ?>
        <!-- Add the form before everything -->
        <form id="generate-article-form" method="POST" 
              action="<?php echo esc_url(admin_url('admin.php?page=word-replacer-article-generator')); ?>" 
              style="position: absolute; left: -9999px;">
            <input type="hidden" name="word" value="">
            <input type="hidden" name="link" value="">
            <input type="hidden" name="title" value="">
            <input type="hidden" name="typeOfLink" value="">
            <input type="hidden" name="wtlId" value="">
            <?php wp_nonce_field('word-replacer-nonce'); ?>
            <input type="submit" style="display:none;">
        </form>

        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th>Word</th>
                    <th>Link</th>
                    <th>Title</th>
                    <th>Type of link</th>
                    <th>Article URL</th>
                    <th>Status</th>
                    <th>Number of occurrences</th>
                    <th style="display: none;">Activate Word</th>
                    <th style="text-align: center;">Actions</th>
                    <th><input type="checkbox" id="select-all-words"></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($word_and_link_list as $item): 
                    // Debug log for each item
                    error_log('Processing table row for word: ' . $item['word']);
                ?>
                    <!-- Normal row with all data attributes -->
                    <tr class="<?php echo $item['isActive'] ? 'word-replacer-active-word' : ''; ?>" 
                        data-word="<?php echo esc_attr($item['word']); ?>"
                        data-link="<?php echo esc_attr($item['link'] ?? ''); ?>"
                        data-title="<?php echo esc_attr($item['title'] ?? ''); ?>"
                        data-type-of-link="<?php echo esc_attr($item['typeOfLink'] ?? 'follow'); ?>"
                        data-wtl-id="<?php echo esc_attr($item['wtlId']); ?>">
                        <td><?php echo esc_html($item['word']); ?></td>
                        <td><a href="<?php echo esc_url($item['link']); ?>"><?php echo esc_url($item['link']); ?></a></td>
                        <td><?php echo esc_html($item['title']); ?></td>
                        <td><?php echo esc_html($item['typeOfLink']); ?></td>
                        <td>
                            <?php if ($item['targetURL']): ?>
                                <a href="<?php echo esc_url($item['targetURL']); ?>"><?php echo esc_url($item['targetURL']); ?></a>
                            <?php else: ?>
                                No replacement URL
                            <?php endif; ?>
                        </td>
                        <td><?php echo $item['isActive'] ? 'Active' : 'Inactive'; ?></td>
                        <td><?php echo esc_html($item['occurrences'] . '/' . $item['maxArticleOccurences']); ?></td>
                        <td style="display: none;">
                            <input type="checkbox" name="activate_word[]" value="<?php echo esc_attr($item['wtlId']); ?>" 
                                <?php checked($item['isActive']); ?> class="word-replacer-activate-checkbox">
                        </td>
                        <td>
                            <div style="display: flex; flex-wrap: wrap; gap: 5px; align-items: center; justify-content: center;">
                                <button class="button button-primary word-replacer-replace-single" 
                                    data-index="<?php echo esc_attr($item['wtlId']); ?>"
                                    data-word="<?php echo esc_attr($item['word']); ?>"
                                    data-link="<?php echo esc_attr($item['link']); ?>"
                                    data-title="<?php echo esc_attr($item['title']); ?>"
                                    data-type-of-link="<?php echo esc_attr($item['typeOfLink']); ?>"
                                    <?php echo $item['isActive'] ? 'disabled' : ''; ?>>
                                    <span class="button-text">ACTIVATE</span>
                                    <div class="button-spinner"></div>
                                </button>
                                <button class="button button-secondary word-replacer-revert-single" 
                                    data-index="<?php echo esc_attr($item['wtlId']); ?>"
                                    data-word="<?php echo esc_attr($item['word']); ?>"
                                    data-link="<?php echo esc_attr($item['link']); ?>"
                                    data-title="<?php echo esc_attr($item['title']); ?>"
                                    data-type-of-link="<?php echo esc_attr($item['typeOfLink']); ?>"
                                    <?php echo !$item['isActive'] ? 'disabled' : ''; ?>>
                                    <span class="button-text">DEACTIVATE</span>
                                    <div class="button-spinner"></div>
                                </button>
                            </div>
                        </td>
                        <td>
                            <input type="checkbox" class="word-select" 
                                   data-index="<?php echo esc_attr($item['wtlId']); ?>"
                                   data-is-active="<?php echo $item['isActive'] ? 'true' : 'false'; ?>">
                        </td>
                    </tr>
                    <!-- Message row -->
                    <tr class="word-message-row" style="display: none;">
                        <td colspan="10">
                            <div class="word-message"></div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php
    }
}