console.log('admin.js loaded');

jQuery(document).ready(function($) {
    console.log("Document ready");
    console.log("jQuery version:", $.fn.jquery);
    console.log("wordReplacerAjax:", wordReplacerAjax);
    
    // Verifică dacă avem toate dependențele necesare
    if (typeof wordReplacerAjax === 'undefined') {
        console.error("wordReplacerAjax is not defined!");
        return;
    }

    if (!wordReplacerAjax.ajax_url) {
        console.error("ajax_url is not defined!");
        return;
    }

    if (!wordReplacerAjax.nonce) {
        console.error("nonce is not defined!");
        return;
    }

    function loadWordLinkTable() {
        let reloadButton = $('#reload-word-link-table');
        let buttonText = reloadButton.find('.button-text');
        let spinner = reloadButton.find('.button-spinner');

        // If the reload button exists, disable it and show spinner
        if (reloadButton.length) {
            reloadButton.prop('disabled', true);
            buttonText.hide();
            spinner.show();
        }

        $.ajax({
            url: wordReplacerAjax.ajax_url,
            type: 'POST',
            data: {
                action: 'get_word_link_table',
                nonce: wordReplacerAjax.nonce
            },
            beforeSend: function() {
                $('#word-link-table').html('<p>Loading...</p>');
            },
            success: function(response) {
                console.log("Word Replacer - AJAX response received", response);
                if (response.success) {
                    $('#word-link-table').html(response.data);
                    attachTableEventHandlers();
                } else {
                    $('#word-link-table').html(
                        '<p>Currently, there are no available subscriptions.</p>' +
                        '<button id="reload-word-link-table" class="button button-primary">' +
                            '<span class="button-text">Reload</span>' +
                            '<div class="button-spinner"></div>' +
                        '</button>'
                    );
                }
            },
            error: function(jqXHR, textStatus, errorThrown) {
                $('#word-link-table').html(
                    '<p>Currently, there are no available subscriptions.</p>' +
                    '<button id="reload-word-link-table" class="button button-primary">' +
                        '<span class="button-text">Reload</span>' +
                        '<div class="button-spinner"></div>' +
                    '</button>'
                );
            },
            complete: function() {
                // Restore button state if it exists
                if (reloadButton.length) {
                    reloadButton.prop('disabled', false);
                    buttonText.show();
                    spinner.hide();
                }
                // Reattach event listener for dynamically re-added button
                $('#reload-word-link-table').off('click').on('click', function() {
                    loadWordLinkTable();
                });
            }
        });
    }

    // Verifică dacă elementul există și încarcă tabelul
    if ($('#word-link-table').length > 0) {
        loadWordLinkTable();
    }

    $('#verify-secret-key').on('click', function(e) {
        e.preventDefault();
        var secretKey = $('#secret-key').val();

        $.ajax({
            url: wordReplacerAjax.ajax_url,
            type: 'POST',
            data: {
                action: 'verify_secret_key',
                nonce: wordReplacerAjax.nonce,
                secret_key: secretKey
            },
            success: function(response) {
                if (response.success) {
                    $('#secret-key-section').hide();
                    $('#settings-content').show();
                    loadWordLinkTable();
                    window.location.reload();
                } else {
                    alert('Error: ' + response.data);
                }
            },
            error: function() {
                alert('An error occurred. Please try again.');
            }
        });
    });

    $('#logout-button').on('click', function(e) {
        e.preventDefault();
        $.ajax({
            url: wordReplacerAjax.ajax_url,
            type: 'POST',
            data: {
                action: 'logout_word_replacer',
                nonce: wordReplacerAjax.nonce
            },
            success: function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert('Error logging out. Please try again.');
                }
            },
            error: function() {
                alert('An error occurred while logging out. Please try again.');
            }
        });
    });

    // Plugin update handler
    $('#update-plugin').on('click', function(e) {
        e.preventDefault();
        console.log('Update button clicked');
        $.ajax({
            url: wordReplacerAjax.ajax_url,
            type: 'POST',
            data: {
                action: 'update_my_plugin',
                nonce: wordReplacerAjax.nonce
            },
            success: function(response) {
                alert("Plugin updated! You need to reactivate the plugin!");
                window.location.reload();
            },
            error: function() {
                alert('Update failed. Please try again.');
                window.location.reload();
            }
        });
    });

    // Function to attach event handlers to table elements
    function attachTableEventHandlers() {
        // Select all checkbox functionality
        $('#select-all-words').off('change').on('change', function() {
            $('.word-select').prop('checked', $(this).prop('checked'));
        });

        // Single word replacement handlers
        $('.word-replacer-replace-single').off('click').on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            var $row = $button.closest('tr');
            
            showSpinner($button);
            disableAllButtons();

            $.ajax({
                url: wordReplacerAjax.ajax_url,
                type: 'POST',
                data: {
                    action: 'replace_content',
                    nonce: wordReplacerAjax.nonce,
                    active_words: [$(this).data('index')],
                    single_word_index: $(this).data('index')
                },
                success: function(response) {
                    handleSingleWordResponse(response, $button, $row);
                },
                error: function() {
                    showNotice('An error occurred while replacing content.', 'error');
                },
                complete: function() {
                    hideSpinner($button);
                    resetButtonStates();
                }
            });
        });

        // Single word revert handlers
        $('.word-replacer-revert-single').off('click').on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            var $row = $button.closest('tr');
            
            showSpinner($button);
            disableAllButtons();

            $.ajax({
                url: wordReplacerAjax.ajax_url,
                type: 'POST',
                data: {
                    action: 'replace_content',
                    nonce: wordReplacerAjax.nonce,
                    active_words: [$(this).data('index')],
                    single_word_index: $(this).data('index'),
                    is_bulk_revert: 'true'
                },
                success: function(response) {
                    handleSingleRevertResponse(response, $button);
                },
                error: function() {
                    showNotice('An error occurred while reverting content.', 'error');
                },
                complete: function() {
                    hideSpinner($button);
                    resetButtonStates();
                }
            });
        });

        // Bulk action handlers
        $('#bulk-replace').off('click').on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            var activeWords = [];
            
            disableAllButtons();
            $('.word-replacer-replace-single, .word-replacer-revert-single').prop('disabled', true);
            
            $('.word-select:checked').each(function() {
                var $row = $(this).closest('tr');
                var index = $(this).data('index');
                if (!$row.hasClass('word-replacer-active-word')) {
                    activeWords.push(index);
                }
            });

            if (activeWords.length === 0) {
                showNotice('Please select at least one inactive word to replace.', 'warning');
                enableButtonsBasedOnState();
                return;
            }

            showSpinner($button);

            $.ajax({
                url: wordReplacerAjax.ajax_url,
                type: 'POST',
                data: {
                    action: 'replace_content',
                    nonce: wordReplacerAjax.nonce,
                    active_words: activeWords
                },
                success: function(response) {
                    if (response.success) {
                        const messages = response.data.split('<br>');
                        messages.forEach(msg => {
                            const wordMatch = msg.match(/['"]([^'"]+)['"]/);
                            if (wordMatch && wordMatch[1]) {
                                const word = wordMatch[1];
                                
                                const isError = !msg.includes('Content replaced');
                                
                                showMessageForWord(word, msg, isError ? 'error' : 'success');
                                
                                const $row = $(`tr[data-word="${word}"]`);
                                if (!isError && msg.includes('Content replaced')) {
                                    updateRowAfterSuccess($row, msg);
                                }
                            }
                        });
                    } else {
                        showNotice(response.data, 'error');
                    }
                },
                error: function() {
                    showNotice('An error occurred while replacing content.', 'error');
                },
                complete: function() {
                    hideSpinner($button);
                    enableButtonsBasedOnState();
                    $('.word-select, #select-all-words').prop('checked', false);
                }
            });
        });

        $('#bulk-revert').off('click').on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            var activeWords = [];
            
            disableAllButtons();
            $('.word-replacer-replace-single, .word-replacer-revert-single').prop('disabled', true);
            
            $('.word-select:checked').each(function() {
                var $row = $(this).closest('tr');
                var index = $(this).data('index');
                if ($row.hasClass('word-replacer-active-word')) {
                    activeWords.push(index);
                }
            });

            if (activeWords.length === 0) {
                showNotice('Please select at least one active word to revert.', 'warning');
                enableButtonsBasedOnState();
                return;
            }

            showSpinner($button);

            $.ajax({
                url: wordReplacerAjax.ajax_url,
                type: 'POST',
                data: {
                    action: 'replace_content',
                    nonce: wordReplacerAjax.nonce,
                    active_words: activeWords,
                    is_bulk_revert: 'true'
                },
                success: function(response) {
                    if (response.success) {
                        handleBulkRevertResponse(response);
                    } else {
                        showNotice(response.data, 'error');
                    }
                },
                error: function() {
                    showNotice('An error occurred while reverting content.', 'error');
                },
                complete: function() {
                    hideSpinner($button);
                    enableButtonsBasedOnState();
                    $('.word-select, #select-all-words').prop('checked', false);
                }
            });
        });
    }

    // Helper functions
    function showSpinner($button) {
        $button.addClass('loading');
        $button.find('.button-spinner').show();
        $button.find('.button-text').hide();
    }

    function hideSpinner($button) {
        $button.removeClass('loading');
        $button.find('.button-spinner').hide();
        $button.find('.button-text').show();
    }

    function showNotice(message, type = 'warning') {
        const $notices = $('#word-replacer-notices');
        const $notice = $('<div>')
            .addClass(`word-replacer-notice notice-${type}`)
            .text(message);
        
        $notices.empty().append($notice);
        
        setTimeout(() => {
            $notice.fadeOut(() => $notice.remove());
        }, 3000);
    }

    function resetButtonStates() {
        $('.word-replacer-replace-single, .word-replacer-revert-single').each(function() {
            var $button = $(this);
            var $row = $button.closest('tr');
            var isRowActive = $row.hasClass('word-replacer-active-word');
            
            if ($button.hasClass('word-replacer-replace-single')) {
                $button.prop('disabled', isRowActive);
            } else if ($button.hasClass('word-replacer-revert-single')) {
                $button.prop('disabled', !isRowActive);
            }
        });
        
        $('#bulk-replace, #bulk-revert').prop('disabled', false);
    }

    function enableButtonsBasedOnState() {
        $('.word-replacer-replace-single, .word-replacer-revert-single').each(function() {
            var $button = $(this);
            var $row = $button.closest('tr');
            var isRowActive = $row.hasClass('word-replacer-active-word');
            
            if ($button.hasClass('word-replacer-replace-single')) {
                $button.prop('disabled', isRowActive);
            } else if ($button.hasClass('word-replacer-revert-single')) {
                $button.prop('disabled', !isRowActive);
            }
        });
        
        $('#bulk-replace, #bulk-revert').prop('disabled', false);
    }

    function disableAllButtons() {
        $('.word-replacer-replace-single, .word-replacer-revert-single, #bulk-replace, #bulk-revert').prop('disabled', true);
    }

    function showMessageForWord(word, message, type = 'success') {
        const $wordRow = $(`tr[data-word="${word}"]`);
        const $messageRow = $wordRow.next('.word-message-row');
        
        // If no message row exists, clone the template and insert after the word row
        if (!$messageRow.length) {
            const $template = $('.word-message-row').first().clone();
            $template.insertAfter($wordRow);
            $messageRow = $wordRow.next('.word-message-row');
        }
        
        const $messageDiv = $messageRow.find('.word-message');
        
        $messageDiv
            .removeClass('success error warning')
            .addClass(type)
            .html(message);
        
        $messageRow.show();
        
        // Add Generate Article button for "No articles found" error
        if (type === 'error') {
            console.log('Adding generate article button for:', word);
            // Get the main row data
            const $mainRow = $(`tr[data-word="${word}"]:not(.word-message-row)`);
            const generateButton = `<button type="button" class="button generate-article" 
                data-word="${$mainRow.data('word')}"
                data-link="${$mainRow.data('link')}"
                data-title="${$mainRow.data('title')}"
                data-type-of-link="${$mainRow.data('type-of-link')}"
                data-wtl-id="${$mainRow.data('wtlId')}">
                Generate Article
            </button>`;
            $messageDiv.append(generateButton);
            
            console.log('Generate button added with data:', {
                word: $mainRow.data('word'),
                link: $mainRow.data('link'),
                title: $mainRow.data('title'),
                typeOfLink: $mainRow.data('type-of-link'),
                wtlId: $mainRow.data('wtlId')
            });
        } else {
            setTimeout(() => {
                $messageDiv.fadeOut(() => {
                    $messageRow.hide();
                });
            }, 15000);
        }
    }

    // Update the word message click handler
    $(document).on('click', '.generate-article', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        console.log('Generate article button clicked');
        
        // Get data directly from the button's data attributes
        const word = $(this).data('word');
        const link = $(this).data('link');
        const title = $(this).data('title');
        const typeOfLink = $(this).data('type-of-link');
        const wtlId = $(this).data('wtl-id');
        
        console.log('Button data attributes:', { word, link, title, typeOfLink, wtlId });
        
        // Create and submit a temporary form
        const $tempForm = $('<form>', {
            method: 'POST',
            action: wordReplacerAjax.admin_url + '?page=word-replacer-article-generator',
            style: 'display: none'
        });
        
        // Add hidden inputs
        $('<input>').attr({
            type: 'hidden',
            name: 'word',
            value: word
        }).appendTo($tempForm);
        
        $('<input>').attr({
            type: 'hidden',
            name: 'link',
            value: link
        }).appendTo($tempForm);
        
        $('<input>').attr({
            type: 'hidden',
            name: 'title',
            value: title
        }).appendTo($tempForm);
        
        $('<input>').attr({
            type: 'hidden',
            name: 'typeOfLink',
            value: typeOfLink
        }).appendTo($tempForm);
        
        $('<input>').attr({
            type: 'hidden',
            name: 'wtlId',
            value: wtlId
        }).appendTo($tempForm);
        
        // Add nonce field
        $('<input>').attr({
            type: 'hidden',
            name: '_wpnonce',
            value: wordReplacerAjax.nonce
        }).appendTo($tempForm);
        
        console.log('Submitting form to:', $tempForm.attr('action'));
        console.log('Form data:', {
            word: word,
            link: link,
            title: title,
            typeOfLink: typeOfLink,
            wtlId: wtlId
        });
        
        // Add to body and submit
        $tempForm.appendTo('body').submit();
    });

    function updateRowAfterSuccess($row, msg) {
        $row.addClass('word-replacer-active-word');
        $row.find('.word-replacer-replace-single').prop('disabled', true);
        $row.find('.word-replacer-revert-single').prop('disabled', false);
        
        const occurrencesMatch = msg.match(/Occurrences: (\d+)/);
        const occurrences = occurrencesMatch ? occurrencesMatch[1] : 0;
        const maxOccurrences = $row.data('maxArticleOccurences') || 0;
        $row.find('td:nth-child(7)').text(`${occurrences}/${maxOccurrences}`);
        
        if (msg.includes('target_url:')) {
            const urlMatch = msg.match(/target_url:\s*([^\s,]+)/);
            if (urlMatch && urlMatch[1]) {
                const articleUrl = urlMatch[1];
                $row.find('td:nth-child(5)').html(`<a href="${articleUrl}">${articleUrl}</a>`);
            }
        }
        $row.find('td:nth-child(6)').text('Active');

        // Folosim data-replaced-keyword din atributul row-ului
        const replacedWord = $row.data('replaced-keyword');
        if (replacedWord) {
            const $wordCell = $row.find('td:first-child');
            const originalWord = $wordCell.contents().first().text().trim();
            
            // Only show replaced word if it's different from the original
            if (replacedWord !== originalWord) {
                if ($wordCell.find('.replaced-word').length === 0) {
                    $wordCell.append($('<div>').addClass('replaced-word').text(`(${replacedWord})`));
                } else {
                    $wordCell.find('.replaced-word').text(`(${replacedWord})`);
                }
            }
        }
    }

    function handleBulkRevertResponse(response) {
        const messages = response.data.split('<br>');
        messages.forEach(msg => {
            const wordMatch = msg.match(/['"]([^'"]+)['"]/);
            if (wordMatch && wordMatch[1]) {
                const word = wordMatch[1];
                const isError = msg.includes('Failed');
                showMessageForWord(word, msg, isError ? 'error' : 'success');
                
                if (!isError) {
                    const $row = $(`tr[data-word="${word}"]`);
                    $row.removeClass('word-replacer-active-word');
                    $row.find('.word-replacer-replace-single').prop('disabled', false);
                    $row.find('.word-replacer-revert-single').prop('disabled', true);
                    $row.find('td:nth-child(6)').text('Inactive');
                    $row.find('td:nth-child(5)').text('No replacement URL');
                    
                    // Remove the replaced word display
                    $row.find('.replaced-word').remove();
                    
                    // Update occurrences count
                    const $occurrencesCell = $row.find('td:nth-child(7)');
                    const [current, max] = $occurrencesCell.text().split('/');
                    $occurrencesCell.text(`${Math.max(0, parseInt(current) - 1)}/${max}`);
                }
            }
        });
    }

    // Add back the handleReplacementError function
    function handleReplacementError(word, errorCode) {
        if (errorCode === 'NO_MATCH' || errorCode === 'NO_ARTICLES') {
            return `<button class="button generate-article" data-word="${word}">
                Generate Article
            </button>`;
        }
        return '';
    }

    // Update handleSingleWordResponse to use handleReplacementError
    function handleSingleWordResponse(response, $button, $row) {
        if (response.success) {
            const messages = response.data.split('<br>');
            messages.forEach(msg => {
                if (msg.includes($button.data('word'))) {
                    const isError = msg.includes('Failed') || 
                                  msg.includes('cannot be activated') || 
                                  msg.includes('Could not activate') ||
                                  msg.includes('No articles found') ||
                                  !msg.includes('Content replaced');
                    
                    showMessageForWord($button.data('word'), msg, isError ? 'error' : 'success');
                    
                    if (!isError && msg.includes('Content replaced')) {
                        updateRowAfterSuccess($row, msg);
                    } else if (isError) {
                        // Add Generate Article button when replacement fails
                        const word = $button.data('word');
                        const errorButton = handleReplacementError(word, 'NO_ARTICLES');
                        if (errorButton) {
                            $row.find('.word-message').append(errorButton);
                        }
                    }
                }
            });
        } else {
            showNotice(response.data, 'error');
        }
    }

    function handleSingleRevertResponse(response, $button) {
        if (response.success) {
            const messages = response.data.split('<br>');
            messages.forEach(msg => {
                if (msg.includes($button.data('word'))) {
                    const isError = msg.includes('Failed');
                    showMessageForWord($button.data('word'), msg, isError ? 'error' : 'success');
                    
                    if (!isError) {
                        const $row = $button.closest('tr');
                        $row.removeClass('word-replacer-active-word');
                        $row.find('.word-replacer-replace-single').prop('disabled', false);
                        $row.find('.word-replacer-revert-single').prop('disabled', true);
                        $row.find('td:nth-child(6)').text('Inactive');
                        $row.find('td:nth-child(5)').text('No replacement URL');
                        
                        // Remove the replaced word display
                        $row.find('.replaced-word').remove();
                        
                        // Update occurrences count
                        const $occurrencesCell = $row.find('td:nth-child(7)');
                        const [current, max] = $occurrencesCell.text().split('/');
                        $occurrencesCell.text(`${Math.max(0, parseInt(current) - 1)}/${max}`);
                    }
                }
            });
        } else {
            showNotice(response.data, 'error');
        }
    }

    // Add view toggle functionality
    $(document).on('click', '.view-mode-btn', function() {
        const mode = $(this).data('mode');
        $('.view-mode-btn').removeClass('active');
        $(this).addClass('active');
        
        if (mode === 'html') {
            $('#article-content-preview').hide();
            $('#article-content-html').show();
        } else {
            $('#article-content-preview').show();
            $('#article-content-html').hide();
        }
    });

    function createDiacriticAlternatives(word) {
        const diacriticMap = {
            'i': '(?:[iî])',
            'n': '(?:[n])',
            'c': '(?:[cç])',
            'h': '(?:[h])',
            'r': '(?:[r])',
            'e': '(?:[eéêë])',
            'a': '(?:[aăâ])',
            'u': '(?:[u])',
            't': '(?:[tț])',
            'o': '(?:[oóôö])',
            's': '(?:[sș])',
            'î': '(?:[iî])',
            'ă': '(?:[aăâ])',
            'â': '(?:[aăâ])',
            'ș': '(?:[sș])',
            'ț': '(?:[tț])',
            'I': '(?:[IÎ])',
            'A': '(?:[AĂÂ])',
            'S': '(?:[SȘ])',
            'T': '(?:[TȚ])',
            ' ': '(?:\\s)'
        };

        return Array.from(word).map(char => 
            diacriticMap[char] || char
        ).join('');
    }
});
