jQuery(document).ready(function($) {
    console.log('article-generator.js loaded');

    // Verifică dacă avem toate dependențele necesare
    if (typeof wordReplacerAjax === 'undefined') {
        console.error("wordReplacerAjax is not defined!");
        return;
    }

    if (!wordReplacerAjax.ajax_url) {
        console.error("ajax_url is not defined!");
        return;
    }

    if (!wordReplacerAjax.nonce) {
        console.error("nonce is not defined!");
        return;
    }

    // Define helper functions first
    function highlightKeywordsInContent(content, keyword) {
        try {
            console.log('Highlighting keywords in content:', {
                keyword: keyword,
                contentLength: content.length
            });

            // First try exact match, like in PHP
            const escapedWord = keyword.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            const diacriticAlternatives = createDiacriticAlternatives(keyword);
            const exactRegex = new RegExp(
                `(?<!\\p{L}")(?:${escapedWord}|${diacriticAlternatives})(?!\\p{L}")(?![^<]*>)(?![^<]*(?:</a>|</?h[1-6]>|</?quote>))`,
                'giu'
            );

            let previewContent = content;
            let foundMatches = false;

            // Check if exact match exists
            if (previewContent.match(exactRegex)) {
                foundMatches = true;
                let matchIndex = 0;
                
                previewContent = previewContent.replace(exactRegex, (match, offset) => {
                    matchIndex++;
                    return `<span class="potential-keyword" data-keyword="${match}" data-match-index="${matchIndex}" data-offset="${offset}">${match}</span>`;
                });
            } else {
                // Split the search word into parts and sort by length (descending)
                const wordParts = keyword.trim().split(' ')
                    .filter(part => part.length >= 3) // Only parts with 3 or more characters
                    .sort((a, b) => b.length - a.length); // Sort by length, longest first
                
                // Try each part, starting with the longest
                let bestMatch = null;
                let bestMatchRegex = null;
                
                for (const part of wordParts) {
                    const escapedPart = part.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
                    const diacriticAlternativesPart = createDiacriticAlternatives(part);
                    const partRegex = new RegExp(
                        `(?<!\\p{L}")(?:${escapedPart}|${diacriticAlternativesPart})(?!\\p{L}")(?![^<]*>)(?![^<]*(?:</a>|</?h[1-6]>|</?quote>))`,
                        'giu'
                    );

                    const match = previewContent.match(partRegex);
                    if (match) {
                        // If this is the first match or it's longer than our previous best match
                        if (!bestMatch || match[0].length > bestMatch.length) {
                            bestMatch = match[0];
                            bestMatchRegex = partRegex;
                        }
                    }
                }

                // If we found a match, apply it
                if (bestMatch && bestMatchRegex) {
                    foundMatches = true;
                    let matchIndex = 0;
                    
                    previewContent = previewContent.replace(bestMatchRegex, (match, offset) => {
                        matchIndex++;
                        return `<span class="potential-keyword" data-keyword="${match}" data-match-index="${matchIndex}" data-offset="${offset}">${match}</span>`;
                    });
                }
            }

            return {
                previewContent: previewContent,
                htmlContent: content,
                foundMatches: foundMatches
            };
        } catch (e) {
            console.error('Error highlighting keywords:', e);
            return {
                previewContent: content,
                htmlContent: content,
                foundMatches: false
            };
        }
    }

    function createDiacriticAlternatives(phrase) {
        const diacriticMap = {
            'a': '(?:[aăâ])',
            'ă': '(?:[aăâ])',
            'â': '(?:[aăâ])',
            'i': '(?:[iî])',
            'î': '(?:[iî])',
            's': '(?:[sș])',
            'ș': '(?:[sș])',
            't': '(?:[tț])',
            'ț': '(?:[tț])',
            'A': '(?:[AĂÂ])',
            'Ă': '(?:[AĂÂ])',
            'Â': '(?:[AĂÂ])',
            'I': '(?:[IÎ])',
            'Î': '(?:[IÎ])',
            'S': '(?:[SȘ])',
            'Ș': '(?:[SȘ])',
            'T': '(?:[TȚ])',
            'Ț': '(?:[TȚ])',
            ' ': '(?:\\s)'
        };

        const diacriticParts = [];
        
        // Process each character in the phrase
        for (let char of phrase) {
            if (diacriticMap[char]) {
                diacriticParts.push(diacriticMap[char]);
            } else {
                // Escape special regex characters
                diacriticParts.push(char.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'));
            }
        }

        return diacriticParts.join('');
    }

    function setFormLoading(isLoading) {
        const $form = $('#article-generator-form');
        const $loadingMessage = $('#form-loading-message');
        const $submitButton = $form.find('button[type="submit"]');
        const $modernLoadingInputs = $form.find('input[name="topic"], input[name="brand"], input[name="industry"], input[name="advertiserDomain"], input[name="language"], select');
        
        if (isLoading) {
            $modernLoadingInputs.addClass('loading-input').prop('disabled', true);
            $submitButton.prop('disabled', true);
            $loadingMessage.show();
        } else {
            $modernLoadingInputs.removeClass('loading-input').prop('disabled', false);
            $submitButton.prop('disabled', false);
            $loadingMessage.hide();
        }
    }

    function populateArticleForm(data) {
        const $form = $('#article-generator-form');
        
        // Populate each field immediately
        if (data.topic) $form.find('#topic').val(data.topic);
        if (data.brand) $form.find('#brand').val(data.brand);
        if (data.industry) $form.find('#industry').val(data.industry);
        if (data.advertiserDomain) $form.find('#advertiser-domain').val(data.advertiserDomain);
        if (data.language) $form.find('#language').val(data.language);
        
        // Remove loading state
        setFormLoading(false);
    }

    function showError(message) {
        const $error = $('#publish-error');
        $error.text(message).show();
        
        // Optional: Hide the error after 5 seconds
        setTimeout(() => {
            $error.fadeOut();
        }, 5000);
    }

    // Initialize form data when page loads
    const $form = $('#article-generator-form');
    if ($form.length) {
        console.log('Form found, initializing...');
        // Show loading state immediately
        setFormLoading(true);
        
        // Get the wtlId from hidden input
        const wtlId = $form.find('input[name="wtlId"]').val();
        console.log('wtlId:', wtlId);
        
        // Make AJAX call to get form information
        $.ajax({
            url: wordReplacerAjax.ajax_url,
            type: 'POST',
            data: {
                action: 'get_article_form_information',
                nonce: wordReplacerAjax.nonce,
                wtlId: wtlId
            },
            success: function(response) {
                console.log('API Response:', response);
                if (response.success && response.data) {
                    populateArticleForm(response.data);
                } else {
                    console.error('Failed to get form information:', response);
                    showError('Failed to load form data');
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX error details:', {
                    status: status,
                    error: error,
                    response: xhr.responseText
                });
                showError('Failed to communicate with the server');
            },
            complete: function() {
                setFormLoading(false); // Remove loading state regardless of success/error
            }
        });
    }

    // View toggle functionality
    $('.view-mode-btn').on('click', function() {
        const mode = $(this).data('mode');
        $('.view-mode-btn').removeClass('active');
        $(this).addClass('active');
        $('.article-view').hide();
        $(`#article-content-${mode}`).show();
    });

    // Article generation form handling
    $('#article-generator-form').on('submit', function(e) {
        e.preventDefault();
        
        const $form = $(this);
        const $submitButton = $form.find('button[type="submit"]');
        const formData = {};
        
        // Collect form data
        $form.find('input, textarea, select').each(function() {
            formData[this.name] = $(this).val();
        });

        // Show loading state
        $submitButton.addClass('generate-loading')
            .prop('disabled', true)
            .text('Generating...');

        // Make AJAX request
        $.ajax({
            url: wordReplacerAjax.ajax_url,
            type: 'POST',
            data: {
                action: 'generate_article',
                nonce: wordReplacerAjax.nonce,
                article_data: formData
            },
            success: function(response) {
                if (response.success && response.data) {
                    handleArticleGeneration(response.data, formData);
                    // Disable generate button after successful generation
                    $submitButton.removeClass('generate-loading')
                        .addClass('generate-disabled')
                        .text('Article Generated')
                        .prop('disabled', true);
                } else {
                    showError(response.data?.error || 'Failed to generate article');
                    // Reset button on error
                    $submitButton.removeClass('generate-loading')
                        .prop('disabled', false)
                        .text('Generate Article');
                }
            },
            error: function() {
                showError('Failed to communicate with the server');
                // Reset button on error
                $submitButton.removeClass('generate-loading')
                    .prop('disabled', false)
                    .text('Generate Article');
            }
        });
    });

    // Handle article generation success
    function handleArticleGeneration(response, formData) {
        if (!response.content || !response.content.htmlContent) {
            console.error('Invalid response format:', response);
            return;
        }

        console.log('Handling article generation with:', {
            response: response,
            formData: formData
        });

        // Get keyword from form data
        const keyword = formData.keyword || '';
        console.log('Using keyword:', keyword);

        // Highlight keywords in content
        const { previewContent, htmlContent, foundMatches } = highlightKeywordsInContent(
            response.content.htmlContent,
            keyword
        );

        console.log('Generated preview content:', previewContent);

        // Update preview
        $('#article-title').html(response.content.title);
        $('#article-content-preview').html(previewContent);
        
        // Update HTML view
        $('#article-content-html textarea').val(htmlContent);

        // Show controls and preview
        $('#article-preview').show();
        $('#article-controls').show();

        // Initially hide publish button until keywords are selected
        $('#publish-article-btn').hide();

        // Attach click handlers to keywords
        attachKeywordHandlers();

        // Ensure keywords are clickable
        $('.potential-keyword').css('cursor', 'pointer');
        
        // Check if we should enable the publish button
        updatePublishButtonState();
    }

    // Add click handler for potential keywords
    function attachKeywordHandlers() {
        $('.potential-keyword').on('click', function() {
            const $this = $(this);
            
            // Get match information
            const matchIndex = $this.data('match-index');
            const offset = $this.data('offset');
            
            console.log('Clicked keyword:', {
                keyword: $this.data('keyword'),
                matchIndex: matchIndex,
                offset: offset
            });
            
            // Store the match information
            $this.data('position', offset);
            
            // Deselect all other keywords first
            $('.potential-keyword').not($this).removeClass('selected-keyword');
            
            // Toggle current keyword
            $this.toggleClass('selected-keyword');
            
            // Update selected count
            const selectedCount = $('.selected-keyword').length;
            $('#selected-count').text(selectedCount);
            
            // Update content based on selection
            if (selectedCount > 0) {
                updateHtmlContent();
            } else {
                // Reset HTML content to original (without links)
                const originalContent = $('#article-content-html textarea').val().replace(/<a[^>]*>(.*?)<\/a>/g, '$1');
                $('#article-content-html textarea').val(originalContent);
            }
            
            // Check if we should enable the publish button
            updatePublishButtonState();
        });
    }

    // New function to update publish button state
    function updatePublishButtonState() {
        const selectedCount = $('.selected-keyword').length;
        const htmlContent = $('#article-content-html textarea').val();
        const hasLinks = /<a\s+(?:[^>]*?\s+)?href=/.test(htmlContent);
        
        // Only show publish button if keywords are selected or links exist
        if (selectedCount > 0 || hasLinks) {
            $('#publish-article-btn').show();
        } else {
            $('#publish-article-btn').hide();
        }
    }

    // Helper function to get text node offset
    function getTextNodeOffset(element) {
        let offset = 0;
        let found = false;
        
        function traverse(node) {
            if (found) return;
            
            if (node === element) {
                found = true;
                return;
            }
            
            if (node.nodeType === 3) { // Text node
                offset += node.length;
            } else {
                for (let child of node.childNodes) {
                    traverse(child);
                }
            }
        }
        
        traverse(document.getElementById('article-content-preview'));
        return offset;
    }

    // Update HTML content using the same logic as PHP
    function updateHtmlContent() {
        const $form = $('#article-generator-form');
        const link = $form.find('input[name="link"]').val();
        const title = $form.find('input[name="title"]').val();
        const typeOfLink = $form.find('input[name="typeOfLink"]').val();
        
        // Get original content without any links
        let htmlContent = $('#article-content-html textarea').val().replace(/<a[^>]*>(.*?)<\/a>/g, '$1');
        
        // Get the selected keyword
        const $selectedKeyword = $('.selected-keyword');
        if (!$selectedKeyword.length) return;

        const keyword = $selectedKeyword.data('keyword');
        const position = $selectedKeyword.data('position');
        
        console.log('Selected keyword:', keyword, 'at position:', position);
        
        // Create replacement function similar to PHP
        const createLink = (match) => {
            let relAttribute = '';
            switch (typeOfLink) {
                case 'NoFollow':
                    relAttribute = ' rel="nofollow"';
                    break;
                case 'UGC':
                    relAttribute = ' rel="ugc"';
                    break;
                case 'Sponsored':
                    relAttribute = ' rel="sponsored"';
                    break;
            }
            return `<a href="${link}"${relAttribute} target="_blank" title="${title}">${match}</a>`;
        };

        // First try exact match, like in PHP
        const escapedWord = keyword.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        const diacriticAlternatives = createDiacriticAlternatives(keyword);
        const exactRegex = new RegExp(
            `(?<!\\p{L}")(?:${escapedWord}|${diacriticAlternatives})(?!\\p{L}")(?![^<]*>)(?![^<]*(?:</a>|</?h[1-6]>|</?quote>))`,
            'giu'
        );

        // Check if exact match exists
        if (htmlContent.match(exactRegex)) {
            let foundPosition = 0;
            let currentPosition = 0;
            
            const callback = (match, offset) => {
                // Check if this is the match at our target position
                if (Math.abs(offset - position) < match.length) {
                    return createLink(match);
                }
                return match;
            };
            
            htmlContent = htmlContent.replace(exactRegex, callback);
        } else {
            // Similar logic for partial matches
            const wordParts = keyword.trim().split(' ');
            
            for (const part of wordParts) {
                if (part.length >= 3) {
                    const escapedPart = part.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
                    const diacriticAlternativesPart = createDiacriticAlternatives(part);
                    const partRegex = new RegExp(
                        `(?<!\\p{L}")(?:${escapedPart}|${diacriticAlternativesPart})(?!\\p{L}")(?![^<]*>)(?![^<]*(?:</a>|</?h[1-6]>|</?quote>))`,
                        'giu'
                    );

                    if (htmlContent.match(partRegex)) {
                        const callback = (match, offset) => {
                            if (Math.abs(offset - position) < match.length) {
                                return createLink(match);
                            }
                            return match;
                        };

                        htmlContent = htmlContent.replace(partRegex, callback);
                        break;
                    }
                }
            }
        }
        
        // Update HTML view
        $('#article-content-html textarea').val(htmlContent);
    }

    // Publish article functionality
    $('#publish-article-btn').on('click', function() {
        const $btn = $(this);
        const htmlContent = $('#article-content-html textarea').val();
        const hasLinks = /<a\s+(?:[^>]*?\s+)?href=/.test(htmlContent);
        
        // Double-check that there are links in the content
        if (!hasLinks) {
            showError('Cannot publish article without any links. Please select at least one keyword.');
            return;
        }
        
        const title = $('#article-title').text();
        const wtlId = $('#article-generator-form input[name="wtlId"]').val();

        $btn.prop('disabled', true).addClass('loading');

        $.ajax({
            url: wordReplacerAjax.ajax_url,
            type: 'POST',
            data: {
                action: 'publish_generated_article',
                nonce: wordReplacerAjax.nonce,
                content: htmlContent,
                title: title,
                wtlId: wtlId
            },
            success: function(response) {
                if (response.success) {
                    window.location.href = response.data.editUrl;
                } else {
                    showError(response.data?.error || 'Failed to publish article');
                }
            },
            error: function() {
                showError('Failed to communicate with the server');
            },
            complete: function() {
                $btn.prop('disabled', false).removeClass('loading');
            }
        });
    });
}); 