<?php
/*
Plugin Name: Adverlink
Plugin URI:  https://adverlink.net/
Description: A plugin to replace specific words in content with hyperlinks, based on a URL provided by an API, excluding headers and links.
Version:     BETA 1.4.4.5
Author:      Aysa.ai 
License:     GPL2
*/

if (!defined('ABSPATH')) {
    exit;
}

define('WORD_REPLACER_VERSION', '1.4.4.5');
define('WORD_REPLACER_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WORD_REPLACER_PLUGIN_URL', plugin_dir_url(__FILE__));

if (!defined('WORD_REPLACER_PLUGIN_FILE')) {
    define('WORD_REPLACER_PLUGIN_FILE', __FILE__);
}

require_once WORD_REPLACER_PLUGIN_DIR . 'includes/class-word-replacer-api.php';
require_once WORD_REPLACER_PLUGIN_DIR . 'includes/class-word-replacer-db.php';
require_once WORD_REPLACER_PLUGIN_DIR . 'includes/class-word-replacer-content-processor.php';
require_once WORD_REPLACER_PLUGIN_DIR . 'includes/class-word-replacer-article-generator.php';
require_once WORD_REPLACER_PLUGIN_DIR . 'admin/class-word-replacer-admin.php';
require_once WORD_REPLACER_PLUGIN_DIR . 'admin/class-word-replacer-settings.php';

function word_replacer_init() {
    $word_replacer_api = new WordReplacerAPI();
    $word_replacer_db = new WordReplacerDB($word_replacer_api);
    $word_replacer_content_processor = new WordReplacerContentProcessor($word_replacer_db, $word_replacer_api);
    $word_replacer_admin = new WordReplacerAdmin($word_replacer_api, $word_replacer_db, $word_replacer_content_processor);

    // Initialize hooks and actions
    $word_replacer_admin->init();

    // Instantiating the WordReplacerSettings class automatically adds the menu and registers the settings
    new WordReplacerSettings($word_replacer_api, $word_replacer_db, $word_replacer_content_processor);
}

add_action('plugins_loaded', 'word_replacer_init');

// Activation hook
register_activation_hook(__FILE__, ['WordReplacerDB', 'create_tables']);
register_activation_hook(__FILE__, 'word_replacer_schedule_cron');

function word_replacer_schedule_cron() {
    if (!wp_next_scheduled('word_replacer_cron_hook')) {
        // Schedule the cron job to run once daily at midnight
        wp_schedule_event(strtotime('tomorrow midnight'), 'daily', 'word_replacer_cron_hook');
    }
}

// Add custom cron schedule
add_filter('cron_schedules', 'word_replacer_add_cron_interval');

function word_replacer_add_cron_interval($schedules) {
    // WordPress already includes a 'daily' interval, so you don't need to define it again.
    return $schedules;
}

// Hook for cron execution
add_action('word_replacer_cron_hook', 'word_replacer_cron_exec');

function word_replacer_cron_exec() {
    // Check if cron is disabled
    if (get_option('word_replacer_disable_cron', true)) {
        error_log('Word Replacer Cron: Cron is disabled in settings');
        return;
    }
    try {
        // Log cron start
        error_log('Word Replacer Cron: Starting execution');

        // Initialize necessary classes
        $word_replacer_api = new WordReplacerAPI();
        $word_replacer_db = new WordReplacerDB($word_replacer_api);
        $word_replacer_content_processor = new WordReplacerContentProcessor($word_replacer_db, $word_replacer_api);
        
        // Get user credentials
        $user_id = get_option('word_replacer_user_id');
        $secret_key = get_option('word_replacer_secret_key');
        error_log("Word Replacer Cron: Using user_id: $user_id");

        $results = [];
        
        // Process word activation and deactivation separately
        $results = array_merge($results, word_replacer_process_activation($word_replacer_api, $word_replacer_db, $word_replacer_content_processor, $user_id, $secret_key));
        $results = array_merge($results, word_replacer_process_deactivation($word_replacer_api, $word_replacer_db, $word_replacer_content_processor, $user_id, $secret_key));

        // Log the final results
        if (!empty($results)) {
            $log_results = array(
                'Source' => 'php-plugin-cron',
                'Level' => 'Info',
                'Action' => 'cron_replacement_results',
                'JsonData' => json_encode($results),
                'UserId' => $user_id,
                'AdditionalInfo' => 'Cron replacement/reversion process completed'
            );
            $word_replacer_api->postLog($log_results);
            error_log('Word Replacer Cron: Completed with results - ' . print_r($results, true));
        }

    } catch (Exception $e) {
        error_log('Word Replacer Cron Error: ' . $e->getMessage());
        error_log('Word Replacer Cron Error Stack Trace: ' . $e->getTraceAsString());
    }
}

function word_replacer_process_activation($word_replacer_api, $word_replacer_db, $word_replacer_content_processor, $user_id, $secret_key) {
    $results = [];
    $max_occurences_article = $word_replacer_api->get_max_occurences($user_id, $secret_key);

    // Process words to be activated
    $words_to_activate = $word_replacer_api->get_word_and_link_for_Replace_Cron($user_id, $secret_key);
    if (is_array($words_to_activate) && !isset($words_to_activate['error'])) {
        foreach ($words_to_activate as $item) {
            if (!isset($item['word']) || !isset($item['link']) || !isset($item['wtlId'])) {
                error_log('Word Replacer Cron: Skipping invalid activation item - missing required fields');
                continue;
            }

            $word = $item['word'];
            $WTLId = $item['wtlId'];
            $was_active = $item['isActive'];

            if ($was_active) {
                error_log("Word Replacer Cron: Skipping '$word' - already active");
                continue;
            }

            // Process word activation
            $api_results = $word_replacer_api->call_search_api($word, parse_url(home_url(), PHP_URL_HOST));
            
            $content_replaced = false;
            $replacement_url = '';

            if (is_array($api_results)) {
                foreach ($api_results as $target_url) {
                    if (!is_string($target_url)) {
                        continue;
                    }

                    $post_id = url_to_postid($target_url);
                    if (!$post_id) {
                        error_log("Word Replacer Cron: Could not find post ID for URL: $target_url");
                        continue;
                    }

                    // Check if max occurrences reached for this URL
                    $current_count = $word_replacer_db->get_replacement_url_count($target_url);
                    if ($current_count >= $max_occurences_article) {
                        error_log("Word Replacer Cron: Max occurrences ($max_occurences_article) reached for URL: $target_url");
                        continue;
                    }

                    // Replace content
                    $result = $word_replacer_content_processor->replace_content_for_post($post_id, $item);
                    if ($result['success'] && $result['count'] > 0) {
                        $count = $result['count'];
                        $message = "Cron: Content replaced for '$word' in post ID: $post_id. Occurrences: $count";
                        error_log("Word Replacer Cron: $message");
                        $results[] = $message;
                        $replacement_url = $target_url;
                        $content_replaced = true;

                        // Update databases
                        $item['replacement_url'] = $replacement_url;
                        $item['occurrences'] = $count;
                        $word_replacer_db->update_single_active_word($item);
                        $word_replacer_api->update_word_to_link($WTLId, $replacement_url, true);
                        break;
                    }
                }
            }

            if (!$content_replaced) {
                $message = "Cron: No suitable replacement found for '$word'";
                error_log("Word Replacer Cron: $message");
                $results[] = $message;
            }
        }
    }

    return $results;
}

function word_replacer_process_deactivation($word_replacer_api, $word_replacer_db, $word_replacer_content_processor, $user_id, $secret_key) {
    $results = [];

    // Process words to be deactivated
    $words_to_deactivate = $word_replacer_api->get_word_and_link_for_Revert_Cron($user_id, $secret_key);
    if (is_array($words_to_deactivate) && !isset($words_to_deactivate['error'])) {
        foreach ($words_to_deactivate as $item) {
            if (!isset($item['word']) || !isset($item['link']) || !isset($item['wtlId'])) {
                error_log('Word Replacer Cron: Skipping invalid deactivation item - missing required fields');
                continue;
            }

            $word = $item['word'];
            $WTLId = $item['wtlId'];
            $was_active = $item['isActive'];

            if (!$was_active) {
                error_log("Word Replacer Cron: Skipping '$word' - already inactive");
                continue;
            }

            // Process word deactivation
            $post_id = url_to_postid($item['targetURL']);
            if (!empty($post_id)) {
                $revert_success = true;
                $result = $word_replacer_content_processor->revert_content_for_post($post_id, $word, $item['link']);
                if (!$result) {
                    $revert_success = false;
                    $message = "Failed to revert content for '$word' in post ID: $post_id";
                    error_log("Word Replacer Cron: $message");
                    $results[] = $message;
                }
                if ($revert_success) {
                    // Update databases
                    $word_replacer_db->deactivate_single_word($word, $item['link']);
                    $word_replacer_api->update_word_to_link($WTLId, '', false);
                    $message = "Cron: Successfully deactivated '$word'";
                    error_log("Word Replacer Cron: $message");
                    $results[] = $message;
                }
            }
        }
    }

    return $results;
}

// Deactivation hook
register_deactivation_hook(__FILE__, 'word_replacer_deactivate');

// Add after the existing cron registration
register_activation_hook(__FILE__, 'word_replacer_schedule_canceled_subscription_cron');

add_filter('cron_schedules', 'word_replacer_add_one_second_cron_interval');

function word_replacer_add_one_second_cron_interval($schedules) {
    $schedules['every_second'] = array(
        'interval' => 1800, // 1 second
        'display'  => __('Every Second')
    );
    return $schedules;
}

function word_replacer_schedule_canceled_subscription_cron() {
    if (!wp_next_scheduled('word_replacer_canceled_subscription_cron_hook')) {
        wp_schedule_event(time(), 'every_second', 'word_replacer_canceled_subscription_cron_hook');
    }
}

// Hook for canceled subscription cron execution
add_action('word_replacer_canceled_subscription_cron_hook', 'word_replacer_canceled_subscription_cron_exec');

function word_replacer_canceled_subscription_cron_exec() {
    // Remove the cron settings check since this should always run
    try {
        // Log cron start
        error_log('Word Replacer Canceled Subscription Cron: Starting execution');

        // Initialize necessary classes
        $word_replacer_api = new WordReplacerAPI();
        $word_replacer_db = new WordReplacerDB($word_replacer_api);
        $word_replacer_content_processor = new WordReplacerContentProcessor($word_replacer_db, $word_replacer_api);
        
        // Get user credentials
        $user_id = get_option('word_replacer_user_id');
        $secret_key = get_option('word_replacer_secret_key');
        error_log("Word Replacer Canceled Subscription Cron: Using user_id: $user_id");

        $results = word_replacer_process_canceled_subscription(
            $word_replacer_api, 
            $word_replacer_db, 
            $word_replacer_content_processor, 
            $user_id, 
            $secret_key
        );

        // Log the final results
        if (!empty($results)) {
            $log_results = array(
                'Source' => 'php-plugin-cron',
                'Level' => 'Info',
                'Action' => 'canceled_subscription_cron_results',
                'JsonData' => json_encode($results),
                'UserId' => $user_id,
                'AdditionalInfo' => 'Canceled subscription cron process completed'
            );
            $word_replacer_api->postLog($log_results);
            error_log('Word Replacer Canceled Subscription Cron: Completed with results - ' . print_r($results, true));
        }

    } catch (Exception $e) {
        error_log('Word Replacer Canceled Subscription Cron Error: ' . $e->getMessage());
        error_log('Word Replacer Canceled Subscription Cron Error Stack Trace: ' . $e->getTraceAsString());
    }
}

function word_replacer_process_canceled_subscription($word_replacer_api, $word_replacer_db, $word_replacer_content_processor, $user_id, $secret_key) {
    $results = [];

    // Get words to be deactivated due to canceled subscription
    $words_to_deactivate = $word_replacer_api->get_word_and_link_for_canceled_subscription_cron($user_id, $secret_key);
    if (is_array($words_to_deactivate) && !isset($words_to_deactivate['error'])) {
        foreach ($words_to_deactivate as $item) {
            if (!isset($item['word']) || !isset($item['link']) || !isset($item['wtlId'])) {
                error_log('Word Replacer Canceled Subscription Cron: Skipping invalid item - missing required fields');
                continue;
            }

            $word = $item['word'];
            $WTLId = $item['wtlId'];
            $was_active = $item['isActive'];

            if (!$was_active) {
                error_log("Word Replacer Canceled Subscription Cron: Skipping '$word' - already inactive");
                continue;
            }

            // Process word deactivation
            $post_id = url_to_postid($item['targetURL']);

            if (!empty($post_id)) {
                $revert_success = true;
                $result = $word_replacer_content_processor->revert_content_for_post($post_id, $word, $item['link']);
                error_log('Result: ' . print_r($result, true));
                if (!$result) {
                    $revert_success = false;
                    $message = "Failed to revert content for '$word' in post ID: $post_id";
                    error_log("Word Replacer Canceled Subscription Cron: $message");
                    $results[] = $message;
                }
                if ($revert_success) {
                    // Update databases
                    $word_replacer_db->deactivate_single_word($word, $item['link']);
                    $word_replacer_api->update_word_to_link($WTLId, '', false);
                    $message = "Canceled Subscription Cron: Successfully deactivated '$word'";
                    error_log("Word Replacer Canceled Subscription Cron: $message");
                    $results[] = $message;
                }
            }
        }
    }

    return $results;
}

// Update deactivation hook to clear the new cron as well
function word_replacer_deactivate() {
    // Curăță cron-urile
    wp_clear_scheduled_hook('word_replacer_cron_hook');
    wp_clear_scheduled_hook('word_replacer_canceled_subscription_cron_hook');
}

// Înlocuim cele două filtre cu unul singur și adăugăm verificare pentru frecvență
function check_for_plugin_update($transient) {
    // Verificăm dacă am făcut deja o verificare în ultimele 12 ore
    $last_check = get_option('adverlink_last_update_check');
    $now = time();
    
    if ($last_check && ($now - $last_check) < 43200) { // 12 ore în secunde
        return $transient;
    }
    
    error_log('Adverlink: Starting update check');
    
    // Actualizăm timestamp-ul ultimei verificări
    update_option('adverlink_last_update_check', $now);
    
    // Restul codului de verificare rămâne neschimbat
    if (!is_object($transient)) {
        $transient = new stdClass();
    }
    
    $plugin_slug = plugin_basename(__FILE__);
    $remote_info = get_plugin_metadata_from_api();
    
    if (!$remote_info) {
        return $transient;
    }
    
    // Initialize response array if not set
    if (!isset($transient->response)) {
        $transient->response = array();
    }
    
    // Initialize checked array if not set
    if (!isset($transient->checked)) {
        $transient->checked = array();
    }
    
    // Force version check regardless of plugin state
    if (version_compare(WORD_REPLACER_VERSION, $remote_info->version, '<')) {
        $transient->response[$plugin_slug] = (object) array(
            'slug' => dirname($plugin_slug),
            'new_version' => $remote_info->version,
            'package' => $remote_info->download_url,
            'tested' => $remote_info->tested_up_to,
            'requires' => $remote_info->requires_at_least,
            'url' => $remote_info->homepage
        );
    }

    // Force check for inactive plugins
    $transient->checked[$plugin_slug] = WORD_REPLACER_VERSION;

    return $transient;
}

// Înlocuim cele două filtre cu unul singur
add_filter('site_transient_update_plugins', 'check_for_plugin_update');

function get_plugin_metadata_from_api() {
    $api_url = 'https://adverlinkstaging-frajdnh7cne2dpc7.eastus2-01.azurewebsites.net/api/version/get-plugin-metadata';
    
    // Get update info from API
    $response = wp_remote_get($api_url);
    if (is_wp_error($response)) {
        error_log('Adverlink update check failed: ' . $response->get_error_message());
        return false;
    }

    $remote_info = json_decode(wp_remote_retrieve_body($response));
    if (!$remote_info) {
        error_log('Adverlink: Invalid update metadata received');
        return false;
    }

    return $remote_info;
}

// Adaugă filtrul pentru informațiile despre plugin
add_filter('plugins_api', 'plugin_update_information', 10, 3);

function plugin_update_information($res, $action, $args) {
    $plugin_slug = dirname(plugin_basename(__FILE__));
    
    if ($action !== 'plugin_information' || 
        !isset($args->slug) || 
        $args->slug !== $plugin_slug) {
        return $res;
    }

    $plugin_info = get_plugin_metadata_from_api();
    if (!$plugin_info) {
        return $res;
    }
    
    return (object) array(
        'name' => $plugin_info->name,
        'slug' => $plugin_info->slug,
        'version' => $plugin_info->version,
        'tested' => $plugin_info->tested_up_to,
        'requires' => $plugin_info->requires_at_least,
        'author' => $plugin_info->author,
        'author_profile' => $plugin_info->author_profile,
        'download_link' => $plugin_info->download_url,
        'trunk' => $plugin_info->trunk,
        'requires_php' => $plugin_info->requires_php,
        'last_updated' => $plugin_info->last_updated,
        'sections' => (array) $plugin_info->sections,
        'banners' => (array) $plugin_info->banners
    );
}


// Șterge funcțiile vechi de gestionare a folderelor
// Adaugă doar aceste funcții esențiale pentru actualizare

add_filter('upgrader_pre_install', 'word_replacer_pre_update', 10, 2);
add_filter('upgrader_post_install', 'word_replacer_after_update', 10, 3);
add_action('upgrader_process_complete', 'word_replacer_upgrade_completed', 10, 2);

function word_replacer_pre_update($return, $args) {
    if (isset($args['plugin']) && $args['plugin'] === plugin_basename(__FILE__)) {
        // Salvează starea înainte de actualizare
        $was_active = is_plugin_active(plugin_basename(__FILE__));
        update_option('word_replacer_was_active', $was_active);
        
        // Dezactivează cron-urile temporar
        wp_clear_scheduled_hook('word_replacer_cron_hook');
        wp_clear_scheduled_hook('word_replacer_canceled_subscription_cron_hook');
        
        error_log('Adverlink: Pre-update - Saving state: ' . ($was_active ? 'active' : 'inactive'));
    }
    return $return;
}

function word_replacer_after_update($return, $hook_extra, $result) {
    if (isset($hook_extra['plugin']) && $hook_extra['plugin'] === plugin_basename(__FILE__)) {
        if (get_option('word_replacer_was_active')) {
            // Reactivează plugin-ul
            $plugin_file = plugin_basename(__FILE__);
            $result = activate_plugin($plugin_file);
            
            if (!is_wp_error($result)) {
                // Reprogramează cron-urile
                if (!wp_next_scheduled('word_replacer_cron_hook')) {
                    wp_schedule_event(strtotime('tomorrow midnight'), 'daily', 'word_replacer_cron_hook');
                }
                if (!wp_next_scheduled('word_replacer_canceled_subscription_cron_hook')) {
                    wp_schedule_event(time(), 'every_second', 'word_replacer_canceled_subscription_cron_hook');
                }
                
                delete_option('word_replacer_was_active');
                error_log('Adverlink: Successfully reactivated after update');
            }
        }
    }
    return $return;
}

function word_replacer_upgrade_completed($upgrader_object, $options) {
    if ($options['action'] === 'update' && $options['type'] === 'plugin') {
        if (isset($options['plugins'])) {
            foreach ($options['plugins'] as $plugin) {
                if ($plugin === plugin_basename(__FILE__)) {
                    // Verificare finală pentru reactivare
                    if (get_option('word_replacer_was_active') && !is_plugin_active($plugin)) {
                        activate_plugin($plugin);
                        delete_option('word_replacer_was_active');
                        error_log('Adverlink: Final activation check completed');
                    }
                }
            }
        }
    }
}
