<?php
class WordReplacerDB {
    private $wpdb;
    private $history_table;
    private $active_words_table;
    private $api;

    public function __construct(WordReplacerAPI $api) {
        global $wpdb;
        $this->wpdb = $wpdb;
        $this->history_table = $this->wpdb->prefix . 'word_replacer_history';
        $this->active_words_table = $this->wpdb->prefix . 'word_replacer_active_words';
        $this->api = $api;
    }

    public function tables_exist() {
        global $wpdb;
    
        // Check if the history table exists
        $history_table_exists = $wpdb->get_var($wpdb->prepare(
            "SHOW TABLES LIKE %s",
            $wpdb->prefix . 'word_replacer_history'
        ));
    
        // Check if the active words table exists
        $active_words_table_exists = $wpdb->get_var($wpdb->prepare(
            "SHOW TABLES LIKE %s",
            $wpdb->prefix . 'word_replacer_active_words'
        ));
    
        // Return true if both tables exist
        return $history_table_exists && $active_words_table_exists;
    }

    //MODIFY ONLY HERE sql columns
    public static function get_table_columns() {
        return [
            'history_table' => [
                'id' => "BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT",
                'post_id' => "BIGINT(20) UNSIGNED NOT NULL",
                'word' => "VARCHAR(190) NOT NULL",
                'link' => "VARCHAR(190) NOT NULL",
                'original_content' => "LONGTEXT NOT NULL",
                'replaced_content' => "LONGTEXT NOT NULL",
                'created_at' => "DATETIME NOT NULL"
            ],
            'active_words_table' => [
                'id' => "BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT",
                'word' => "VARCHAR(190) NOT NULL",
                'link' => "VARCHAR(190) NOT NULL",
                'replacement_url' => "VARCHAR(190) NOT NULL",
                'max_occurrences' => "INT UNSIGNED NOT NULL",
                'occurrences' => "INT UNSIGNED NOT NULL",
                'is_active' => "TINYINT(1) NOT NULL DEFAULT 1",
                'replacedKeyword' => "VARCHAR(190) NULL",
                'created_at' => "DATETIME NOT NULL",
                'updated_at' => "DATETIME NOT NULL"
            ]
        ];
    }

    //new method 
    public static function create_tables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        $tables = self::get_table_columns();
    
        // Table names
        $history_table = $wpdb->prefix . 'word_replacer_history';
        $active_words_table = $wpdb->prefix . 'word_replacer_active_words';
    
        // Instantiate API and Database classes to call get_Plugin_Status
        $word_replacer_api = new WordReplacerAPI();
        $plugin_status = $word_replacer_api->get_Plugin_Status();
    
        // SQL for history table
        $history_table_sql = "CREATE TABLE IF NOT EXISTS $history_table (";
        foreach ($tables['history_table'] as $column => $definition) {
            $history_table_sql .= "$column $definition, ";
        }
        $history_table_sql .= "PRIMARY KEY (id), KEY post_id (post_id), KEY word (word)) $charset_collate;";
    
        // SQL for active words table
        $active_words_table_sql = "CREATE TABLE IF NOT EXISTS $active_words_table (";
        foreach ($tables['active_words_table'] as $column => $definition) {
            $active_words_table_sql .= "$column $definition, ";
        }
        $active_words_table_sql .= "PRIMARY KEY (id), UNIQUE KEY wordlink (word, link)) $charset_collate;";
    
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($history_table_sql);
        dbDelta($active_words_table_sql);
    }
    
    public function alter_tables() {
        global $wpdb;
        $tables = self::get_table_columns();
    
        // Function to sync table columns
        function sync_table_columns($table_name, $expected_columns) {
            global $wpdb;
    
            // Get current columns in the table
            $current_columns = $wpdb->get_col("SHOW COLUMNS FROM $table_name");
    
            // Add missing columns
            foreach ($expected_columns as $column => $definition) {
                if (!in_array($column, $current_columns)) {
                    $wpdb->query("ALTER TABLE $table_name ADD $column $definition;");
                }
            }
    
            // Drop extra columns
            foreach ($current_columns as $column) {
                if (!array_key_exists($column, $expected_columns)) {
                    $wpdb->query("ALTER TABLE $table_name DROP COLUMN $column;");
                }
            }
        }
    
        // Sync history table
        $history_table = $wpdb->prefix . 'word_replacer_history';
        sync_table_columns($history_table, $tables['history_table']);
    
        // Sync active words table
        $active_words_table = $wpdb->prefix . 'word_replacer_active_words';
        sync_table_columns($active_words_table, $tables['active_words_table']);
    }

    public function get_active_words() {
        return $this->wpdb->get_results("SELECT * FROM {$this->active_words_table} WHERE is_active = 1", ARRAY_A);
    }

    public function get_occurrences($word, $link = null) {
        $query = "SELECT occurrences FROM {$this->active_words_table} WHERE word = %s";
        
        if ($link) {
            $query .= " AND link = %s";
            return $this->wpdb->get_var($this->wpdb->prepare($query, $word, $link));
        }
        
        return $this->wpdb->get_var($this->wpdb->prepare($query, $word));
    }


    //deprecated
    public function get_replacement_url($word, $link = null) {
        $query = "SELECT replacement_url FROM {$this->active_words_table} WHERE word = %s";
        
        if ($link) {
            $query .= " AND link = %s"; // Assuming 'link' is a column in your table
            return $this->wpdb->get_var($this->wpdb->prepare($query, $word, $link));
        }
        
        return $this->wpdb->get_var($this->wpdb->prepare($query, $word));
    }

    //deprecated 14.08.2024
    public function update_active_words($words, $active_indices) {
        $this->wpdb->query("UPDATE {$this->active_words_table} SET is_active = 0");
        foreach ($words as $index => $word) {
            $is_active = in_array($index, $active_indices) ? 1 : 0;
            $this->wpdb->replace(
                $this->active_words_table,
                [
                    'word' => $word['word'],
                    'link' => $word['link'],
                    'replacement_url' => $word['replacement_url'] ?? '',
                    'max_occurrences' => isset($word['max_occurrences']) ? intval($word['max_occurrences']) : 1,
                    'occurrences' => isset($word['occurrences']) ? intval($word['occurrences']) : 0,
                    'is_active' => $is_active,
                    'created_at' => current_time('mysql'),
                    'updated_at' => current_time('mysql'),
                ],
                ['%s', '%s', '%s', '%d', '%d', '%d', '%s', '%s']
            );
        }
    }
    public function deactivate_single_word($word, $link) {
        // Log the deactivation attempt
        // $log_data = array(
        //     'word' => $word,
        //     'link' => $link,
        // );
        // $logEntry = array(
        //     'Source' => 'php-plugin',
        //     'Level' => 'Info',
        //     'Action' => 'deactivate_single_word_start',
        //     'JsonData' => json_encode($log_data),
        //     'UserId' => null,
        //     'AdditionalInfo' => 'Deactivating single active word'
        // );
        // $this->api->postLog($logEntry);

        return $this->wpdb->update(
            $this->active_words_table,
            [
                'is_active' => 0,
                'occurrences' => 0,
                'replacement_url' => '',
                'updated_at' => current_time('mysql'),
            ],
            [
                'word' => $word,
                'link' => $link
            ],
            ['%d', '%d', '%s', '%s'],
            ['%s', '%s']
        );
    }

    public function update_single_active_word($word) {
        // Log the update attempt for a single active word
        // $log_data = array(
        //     'word' => $word['word'],
        //     'link' => $word['link'],
        //     'replacement_url' => $word['replacement_url'] ?? '',
        //     'max_occurrences' => isset($word['max_occurrences']) ? intval($word['max_occurrences']) : 1,
        //     'occurrences' => isset($word['occurrences']) ? intval($word['occurrences']) : 0,
        // );
        // $logEntry = array(
        //     'Source' => 'php-plugin',
        //     'Level' => 'Info',
        //     'Action' => 'update_single_active_word_start',
        //     'JsonData' => json_encode($log_data),
        //     'UserId' => null,
        //     'AdditionalInfo' => 'Updating a single active word'
        // );
        // $this->api->postLog($logEntry);

        return $this->wpdb->replace(
            $this->active_words_table,
            [
                'word' => $word['word'],
                'link' => $word['link'],
                'replacement_url' => $word['replacement_url'] ?? '',
                'max_occurrences' => isset($word['max_occurrences']) ? intval($word['max_occurrences']) : 1,
                'occurrences' => isset($word['occurrences']) ? intval($word['occurrences']) : 0,
                'is_active' => 1,
                'replacedKeyword' => $word['replacedKeyword'] ?? null,
                'updated_at' => current_time('mysql'),
            ],
            ['%s', '%s', '%s', '%d', '%d', '%d', '%s', '%s']
        );
    }
    public function get_affected_posts($word, $link) {
        global $wpdb;
        $history_table = $this->history_table;
        
        $query = $wpdb->prepare(
            "SELECT DISTINCT post_id FROM $history_table WHERE word = %s AND link = %s",
            $word,
            $link
        );
        
        return $wpdb->get_col($query);
    }
    
    public function add_history($post_id, $word, $link, $original_content, $replaced_content) {
        // Log the addition of history
        // $log_data = array(
        //     'post_id' => $post_id,
        //     'word' => $word,
        //     'link' => $link,
        //     'original_content' => $original_content,
        //     'replaced_content' => $replaced_content,
        // );
        // $logEntry = array(
        //     'Source' => 'php-plugin',
        //     'Level' => 'Info',
        //     'Action' => 'add_history_start',
        //     'JsonData' => json_encode($log_data),
        //     'UserId' => null,
        //     'AdditionalInfo' => 'Adding history entry'
        // );
        // $this->api->postLog($logEntry);

        $this->wpdb->insert(
            $this->history_table,
            [
                'post_id' => $post_id,
                'word' => $word,
                'link' => $link,
                'original_content' => $original_content,
                'replaced_content' => $replaced_content,
                'created_at' => current_time('mysql'),
            ],
            ['%d', '%s', '%s', '%s', '%s', '%s']
        );
    }
    

    public function get_original_content($post_id, $word, $link) {
        return $this->wpdb->get_var($this->wpdb->prepare(
            "SELECT original_content FROM {$this->history_table} WHERE post_id = %d AND word = %s AND link = %s ORDER BY created_at DESC LIMIT 1",
            $post_id,
            $word,
            $link
        ));
    }
    //deprecated
//COLLATE utf8mb4_bin is important dont remove it!!! actually nvm, we should check on word id. to be added. there are issues if we have similar words (capital letters)
    public function is_word_active($word, $link = null) {
        $query = "SELECT COUNT(*) FROM {$this->active_words_table} WHERE word = %s AND is_active = 1";
        
        if ($link) {
            $query .= " AND link = %s"; // Assuming 'link' is a column in your table
            return (bool) $this->wpdb->get_var($this->wpdb->prepare($query, $word, $link));
        }
        
        return (bool) $this->wpdb->get_var($this->wpdb->prepare($query, $word));
    }

    public function remove_history($post_id, $word, $link) {
        return $this->wpdb->delete(
            $this->history_table,
            [
                'post_id' => $post_id,
                'word' => $word,
                'link' => $link,
            ],
            ['%d', '%s', '%s']
        );
    }

    public function get_replacement_url_count($replacement_url) {
        return $this->wpdb->get_var($this->wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->active_words_table} WHERE replacement_url = %s AND is_active = 1" ,
            $replacement_url
        )) ?: 0;
    }
}